using System;
using System.Text;
using System.Windows.Forms;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.GUI;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// User control encapsulating logic for editing both VistALink and HL7 interface's configurations.
	/// </summary>
	public class EditCombinedInterfaces : BaseAdminFormControl
	{
		private gov.va.med.vbecs.admin.GUI.EditInterfaceParameters editInterfaceParameters;
		private System.Windows.Forms.GroupBox grpSelectInterface;
		private System.Windows.Forms.ListBox lstInterfaces;
		private gov.va.med.vbecs.admin.GUI.EditVistALinkConnectionParameters editVistALinkConnectionParameters;

		/// <summary>
		/// VistALinkInterfaceDisplayIndex
		/// </summary>
		protected const int VistALinkInterfaceDisplayIndex = 0;

		/// <summary>
		/// VistALinkInterfaceDisplayName
		/// </summary>
		protected const string VistALinkInterfaceDisplayName = "VistALink";

		/// <summary>
		/// InterfaceType
		/// </summary>
		protected enum InterfaceType
		{
			/// <summary>
			/// VistALink
			/// </summary>
			VistALink,
			/// <summary>
			/// HL7
			/// </summary>
			HL7
		}
		
		/// <summary>
		/// Abstract Class InterfaceConfiguratorDisplayItem
		/// </summary>
		protected abstract class InterfaceConfiguratorDisplayItem 
		{
			/// <summary>
			/// DisplayMemberName
			/// </summary>
			public const string DisplayMemberName = "InterfaceName";

			/// <summary>
			/// InterfaceType
			/// </summary>
			public readonly InterfaceType InterfaceType;

			/// <summary>
			/// InterfaceConfigurator
			/// </summary>
			public readonly object InterfaceConfigurator;

			/// <summary>
			/// InterfaceName
			/// </summary>
			public abstract string InterfaceName{ get; }

			/// <summary>
			/// Constructor
			/// </summary>
			/// <param name="interfaceType"></param>
			/// <param name="interfaceConfigurator"></param>
			public InterfaceConfiguratorDisplayItem( InterfaceType interfaceType, object interfaceConfigurator )
			{
				InterfaceType = interfaceType;
				InterfaceConfigurator = interfaceConfigurator;
			}
		}

		/// <summary>
		/// Class HL7ConfiguratorDisplayItem
		/// </summary>
		protected class HL7ConfiguratorDisplayItem : InterfaceConfiguratorDisplayItem
		{
			/// <summary>
			/// Constructor
			/// </summary>
			/// <param name="interfaceConfigurator"></param>
			public HL7ConfiguratorDisplayItem( HL7InterfaceConfig interfaceConfigurator ) 
				: base( InterfaceType.HL7, interfaceConfigurator ) {}

			/// <summary>
			/// Get InterfaceName
			/// </summary>
			public override string InterfaceName
			{
				get
				{
					//CR 2940
					return ((HL7InterfaceConfig)InterfaceConfigurator).Interface.InterfaceName.Replace("_", " ");
				}
			}

			/// <summary>
			/// Get IsInterfaceConfigDirty
			/// </summary>
			public bool IsInterfaceConfigDirty
			{
				get
				{
					return ((HL7InterfaceConfig)InterfaceConfigurator).IsDirty;
				}
			}
		}

		/// <summary>
		/// Class VistALinkConfiguratorDisplayItem
		/// </summary>
		protected class VistALinkConfiguratorDisplayItem : InterfaceConfiguratorDisplayItem
		{
			/// <summary>
			/// Constructor
			/// </summary>
			public VistALinkConfiguratorDisplayItem() 
				: base( InterfaceType.VistALink, null ) {}

			/// <summary>
			/// Get InterfaceName
			/// </summary>
			public override string InterfaceName
			{
				get
				{
					return VistALinkInterfaceDisplayName;
				}
			}
		}
		
		/// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		/// <summary>
		/// EditCombinedInterfaces
		/// </summary>
		public EditCombinedInterfaces()
		{
			// This call is required by the Windows.Forms Form Designer
			InitializeComponent();

			lstInterfaces.DisplayMember = InterfaceConfiguratorDisplayItem.DisplayMemberName;
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.editVistALinkConnectionParameters = new gov.va.med.vbecs.admin.GUI.EditVistALinkConnectionParameters();
			this.editInterfaceParameters = new gov.va.med.vbecs.admin.GUI.EditInterfaceParameters();
			this.grpSelectInterface = new System.Windows.Forms.GroupBox();
			this.lstInterfaces = new System.Windows.Forms.ListBox();
			this.grpSelectInterface.SuspendLayout();
			this.SuspendLayout();
			// 
			// editVistALinkConnectionParameters
			// 
			this.editVistALinkConnectionParameters.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.editVistALinkConnectionParameters.Location = new System.Drawing.Point(124, 0);
			this.editVistALinkConnectionParameters.Name = "editVistALinkConnectionParameters";
			this.editVistALinkConnectionParameters.Size = new System.Drawing.Size(468, 448);
			this.editVistALinkConnectionParameters.TabIndex = 14;
			// 
			// editInterfaceParameters
			// 
			this.editInterfaceParameters.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.editInterfaceParameters.Location = new System.Drawing.Point(124, 0);
			this.editInterfaceParameters.Name = "editInterfaceParameters";
			this.editInterfaceParameters.Size = new System.Drawing.Size(468, 448);
			this.editInterfaceParameters.TabIndex = 14;
			// 
			// grpSelectInterface
			// 
			this.grpSelectInterface.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left)));
			this.grpSelectInterface.Controls.Add(this.lstInterfaces);
			this.grpSelectInterface.Location = new System.Drawing.Point(0, 0);
			this.grpSelectInterface.Name = "grpSelectInterface";
			this.grpSelectInterface.Size = new System.Drawing.Size(120, 448);
			this.grpSelectInterface.TabIndex = 13;
			this.grpSelectInterface.TabStop = false;
			this.grpSelectInterface.Text = "Select Interface";
			// 
			// lstInterfaces
			// 
			this.lstInterfaces.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lstInterfaces.Location = new System.Drawing.Point(8, 16);
			this.lstInterfaces.Name = "lstInterfaces";
			this.lstInterfaces.Size = new System.Drawing.Size(104, 407);
			this.lstInterfaces.TabIndex = 0;
			this.lstInterfaces.SelectedIndexChanged += new System.EventHandler(this.lstInterfaces_SelectedIndexChanged);
			// 
			// EditCombinedInterfaces
			// 
			this.Controls.Add(this.editVistALinkConnectionParameters);
			this.Controls.Add(this.editInterfaceParameters);
			this.Controls.Add(this.grpSelectInterface);
			this.Name = "EditCombinedInterfaces";
			this.Size = new System.Drawing.Size(594, 449);
			this.Load += new System.EventHandler(this.EditCombinedInterfaces_Load);
			this.grpSelectInterface.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// lstInterfaces_SelectedIndexChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void lstInterfaces_SelectedIndexChanged(object sender, System.EventArgs e)
		{
#if !DESIGN
			bool _editVistALinkMode = ( ((InterfaceConfiguratorDisplayItem)lstInterfaces.SelectedItem).InterfaceType == InterfaceType.VistALink );
			
			editVistALinkConnectionParameters.Visible = _editVistALinkMode;
			editInterfaceParameters.Visible = !_editVistALinkMode;			

			if( !_editVistALinkMode )
				OnHL7InterfaceSelected();
#endif
		}

		/// <summary>
		/// HL7InterfaceSelected Event
		/// </summary>
		protected void OnHL7InterfaceSelected()
		{
#if !DESIGN
			HL7InterfaceConfig _cfg = (HL7InterfaceConfig)((InterfaceConfiguratorDisplayItem)lstInterfaces.SelectedItem).InterfaceConfigurator;
			editInterfaceParameters.SwitchToInterfaceConfig( _cfg );
#endif
		}

		/// <summary>
		/// PopulateListOfInterfaces
		/// </summary>
		protected void PopulateListOfInterfaces()
		{						
			foreach( HL7InterfaceConfig _interfaceConfig in HL7InterfaceConfig.GetHL7InterfaceConfigs() )
				lstInterfaces.Items.Add( new HL7ConfiguratorDisplayItem( _interfaceConfig ) );

			lstInterfaces.Items.Insert( VistALinkInterfaceDisplayIndex, new VistALinkConfiguratorDisplayItem() );
		}

		/// <summary>
		/// SelectDefaultInterface
		/// </summary>
		protected void SelectDefaultInterface()
		{
			SelectVistaLinkInterface();
		}

		/// <summary>
		/// SelectVistaLinkInterface
		/// </summary>
		protected void SelectVistaLinkInterface()
		{
			lstInterfaces.SelectedIndex = VistALinkInterfaceDisplayIndex;
			lstInterfaces_SelectedIndexChanged( lstInterfaces, EventArgs.Empty );
		}

		/// <summary>
		/// EditCombinedInterfaces_Load
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void EditCombinedInterfaces_Load(object sender, System.EventArgs e)
		{
#if !DESIGN
			PopulateListOfInterfaces();
			SelectDefaultInterface();
#endif
		}

		/// <summary>
		/// Get IsDataUnsaved
		/// </summary>
		protected bool IsDataUnsaved
		{
			get
			{
				return editVistALinkConnectionParameters.IsDataUnsaved || IsHL7InterfacesDataUnsaved;
			}
		}

		/// <summary>
		/// IsHL7InterfacesDataUnsaved
		/// </summary>
		protected bool IsHL7InterfacesDataUnsaved
		{
			get
			{
				foreach( InterfaceConfiguratorDisplayItem _item in lstInterfaces.Items )
				{
					if( !(_item is HL7ConfiguratorDisplayItem ) ) 
						continue;

					if( ((HL7ConfiguratorDisplayItem)_item).IsInterfaceConfigDirty )
						return true;
				}

				return false;
			}
		}

		/// <summary>
		/// SwitchToFirstInterfaceWithUnsavedData
		/// </summary>
		protected void SwitchToFirstInterfaceWithUnsavedData()
		{
			if( editVistALinkConnectionParameters.IsDataUnsaved )
				SelectVistaLinkInterface();
			else
				SelectFirstHl7InterfaceWithUnsavedData();
		}

		/// <summary>
		/// SelectFirstHl7InterfaceWithUnsavedData
		/// </summary>
		protected void SelectFirstHl7InterfaceWithUnsavedData()
		{
			foreach( InterfaceConfiguratorDisplayItem _item in lstInterfaces.Items )
			{
				if( !(_item is HL7ConfiguratorDisplayItem ) || !((HL7ConfiguratorDisplayItem)_item).IsInterfaceConfigDirty ) 
					continue;

				lstInterfaces.SelectedItem = _item;
				lstInterfaces_SelectedIndexChanged( lstInterfaces, EventArgs.Empty );
				break;
			}
		}

		/// <summary>
		/// GetNewLineSeparatedListOfUnsavedInterfaces
		/// </summary>
		/// <returns></returns>
		protected string GetNewLineSeparatedListOfUnsavedInterfaces()
		{
			StringBuilder _sb = new StringBuilder();
			
			_sb.Append( Environment.NewLine );

			if( editVistALinkConnectionParameters.IsDataUnsaved )
			{
				_sb.Append( ((VistALinkConfiguratorDisplayItem)lstInterfaces.Items[VistALinkInterfaceDisplayIndex]).InterfaceName );
				_sb.Append( Environment.NewLine );
			}

			foreach( InterfaceConfiguratorDisplayItem _item in lstInterfaces.Items )
			{
				if( !(_item is HL7ConfiguratorDisplayItem ) || !((HL7ConfiguratorDisplayItem)_item).IsInterfaceConfigDirty ) 
					continue;

				_sb.Append( ((HL7ConfiguratorDisplayItem)_item).InterfaceName );
				_sb.Append( Environment.NewLine );
			}

			_sb.Append( Environment.NewLine );

			return _sb.ToString();
		}

		/// <summary>
		/// ConfirmExit
		/// </summary>
		/// <returns></returns>
		public bool ConfirmExit()
		{
			if( !this.IsDataUnsaved )
				return true;

			if( ResultAfterHostRefresh( GuiMessenger.ShowMessageBox( Common.StrRes.ConfMsg.MUC04.InterfaceDataNotSaved( GetNewLineSeparatedListOfUnsavedInterfaces() ) ) == DialogResult.Yes ) )
				return true;
			
			SwitchToFirstInterfaceWithUnsavedData();

			return false;
		}
	}
}
